# デジタル心易システム仕様 - Digital Shin-eki System

占術とは、時機を読み、流れを観る技術である。
易経六十四卦は、宇宙の変化の法則を体系化した最古の叡智。
「デジタル心易」は、古代の易経とデジタル技術を融合し、
心の問いをデジタル演算によって卦爻に変換する、
21世紀の占断システムである。

---

## 🔮 デジタル心易の基本思想

### 易の本質 - 変化の書
> 「易」とは「変化」を意味する。
> 万物は常に変化し、その変化には法則がある。
> 64卦384爻は、その変化のパターンを網羅的に記述したもの。

### デジタル心易の革新
> 「心易」とは、心の問いに易が応えること。
> デジタル技術により、占的を確定的な数値に変換し、
> 再現性と客観性を持った占断を実現する。

### 至誠の原理
> 「至誠通天」- 至誠は天に通ず
> 「至誠無息」- 至誠は息むことなし
>
> 誠実な問いには誠実な答えが返る。
> デジタル演算は偶然ではなく、
> 問いの本質（ハッシュ値）と天地人の調和（XOR）による必然。

### 占的の明確化 - 問いの技術
**曖昧な問いには曖昧な答えしか返らない**

占的明確化の三原則：
1. **具体性**：抽象的な問いを具体的な選択に変換
2. **時限性**：「いつまでに」を明確にする
3. **行動性**：「どうすべきか」を問う形にする

例：
- ❌ 「仕事運は？」（曖昧）
- ✅ 「3ヶ月以内にA社の転職オファーを受けるべきか」（明確）

---

## 📊 システムアーキテクチャ

### ファイル構造
```
I-Ching/
├── デジタル心易システム仕様.md    # 本仕様書
├── 大卦データベース.json    # 64卦384爻の完全データ
├── iching_divination.py   # 占断計算エンジン
├── DivineTemplate.md      # 占断書テンプレート
└── 変卦仕様_append.md     # 変卦システム拡張仕様
```

### データベース構造（大卦データベース.json）
```json
{
  "metadata": {
    "title": "易経六十四卦384爻データベース",
    "version": "2.0.0",
    "format": "大環主指定フォーマット"
  },
  "hexagrams": [  // 配列形式（0始まり、番号-1でアクセス）
    {
      "番号": 1,  // 1-64の卦番号
      "名前": "乾為天",
      "読み": "けんいてん",
      "バイナリ": "111111",  // 6桁の2進数表現
      "シンボル": "䷀",  // Unicode易経記号
      "卦辞": "元亨利贞。",
      "爻": [  // 6つの爻データの配列
        {
          "番号": 1,  // 1-6の爻番号
          "名前": "初九",
          "バイナリ": 1,  // 0:陰爻, 1:陽爻
          "陰陽": "陽",
          "爻辞": "潜龙，勿用。"
        }
        // ... 他の爻
      ]
    }
    // ... 他の63卦
  ]
}
```

**重要**：
- `hexagrams`は配列なので、20番の卦は`hexagrams[19]`でアクセス
- バイナリ表現での検索には全配列の走査が必要
- 変卦計算時はバイナリ変換後の再検索が必要

### 処理フロー
1. **占的ヒアリング** → 対話による問いの明確化（問いを立てる）
2. **占的確定** → 具体的な文として固定し、背景情報を整理
3. **卦爻算出** → デジタル本筮法による演算
   - BASE64変換（筮竹に託す）→ SHA256（天地創造）→ 三才分割 → XOR統合 → MOD64
   - タイムスタンプから得爻を決定（略筮法的アプローチ）
4. **占断生成** → テンプレートに基づく解釈展開と変卦分析

### データベースアクセスの実装
```python
# iching_divination.pyでの実装
class IChingDivination:
    def __init__(self, database_path=None):
        with open(database_path, 'r', encoding='utf-8') as f:
            self.database = json.load(f)
        self.hexagrams = self.database['hexagrams']  # 配列として保持
    
    def get_hexagram_data(self, hexagram_number: int):
        # 番号は1始まり、配列は0始まりなので-1
        return self.hexagrams[hexagram_number - 1]
    
    # 変卦計算時のバイナリ検索（拡張版）
    def find_hexagram_by_binary(self, binary: str):
        for hexagram in self.hexagrams:
            if hexagram['バイナリ'] == binary:
                return hexagram
        return None
```

---

## 🎲 卦爻決定アルゴリズム

### 本筮法とデジタル心易の対応

古代の本筮法と21世紀のデジタル心易は、同じ易の原理を異なる形で実装したものである。

| 本筮法 | デジタル心易 | 備考 |
|--------|-------------|------|
| 問いを立てる | 占的・背景を入力する | 心の問いを明確化 |
| 問いを筮竹に託す | 占的・背景をBASE64変換する | 64卦との共鳴で易の体系に入る |
| 筮竹を50→49本に調整する | BASE64パディングを行う | 太極を置く（＝記号が太極に相当） |
| 筮竹を二分する | SHA256ハッシュ化する | BASE64→SHA256を天地創造と置く |
| 筮竹を天地人に分ける | SHA256ハッシュを三才分割する | 天(16文字)・地(32文字)・人(16文字) |
| 筮竹を4本ずつ数える | 三才をXOR演算する | 天⊕地⊕人を統合を行う |
| 余りを数える | MOD64の計算をする | 卦を求めるための数理的操作 |
| 爻を重ねて卦を得る | 大卦JSONから卦を得る | 求まった数霊を64卦に変換 |
| 老陽老陰から変爻を得る | TIMESTAMPから爻位を得る | あえて略筮法を使って変卦を用いる |

この対応表が示すように、デジタル心易は本筮法の精神を
デジタル技術で再現した現代の易占システムである。

### デジタル本筮法の実装 - 占的から得卦を導く
```python
import base64
import hashlib

def get_hexagram_number(占的: str, 状況整理: str = "") -> int:
    """占的文字列と状況整理から卦番号（1-64）を決定
    天地人の三才思想に基づく実装"""

    # 至誠通天 - 誠の心が天に通じる
    print("至誠通天 - 誠の心をもって問いを天に届けます")

    # 占的と状況整理を結合（状況が個別性を生む）
    完全な問い = f"{占的}\n===状況整理===\n{状況整理}" if 状況整理 else 占的

    # UTF-8エンコード → BASE64 → ハッシュ化
    encoded = base64.b64encode(完全な問い.encode('utf-8'))
    hash_value = hashlib.sha256(encoded).hexdigest()

    # 天地人の三才に分割（SHA256は64文字）
    天 = int(hash_value[:16], 16)    # 16文字（64ビット - 上界の意志）
    地 = int(hash_value[16:48], 16)  # 32文字（128ビット - 天と人を支える基盤）
    人 = int(hash_value[48:64], 16)  # 16文字（64ビット - 人間の問い）

    # 三才をXOR演算で統合（天地人の調和）
    三才統合 = 天 ^ 地 ^ 人

    # 64卦への変換
    number = 三才統合 % 64 + 1
    return number
```

### デジタル略筮法の実装 - 占機から得爻を導く
```python
import time

def get_line_number() -> int:
    """現在時刻から爻番号（1-6）を決定"""
    # ミリ秒単位のタイムスタンプ
    timestamp = int(time.time() * 1000)
    # mod6で0-5、+1で1-6に変換
    line_number = timestamp % 6 + 1
    return line_number
```

### 設計思想
- **本筮法の継承**：古代の筮竹占いをデジタルで再現
- **BASE64の意義**：64卦との数理的共鳴を生み出す変換
  - パディング記号（＝）は太極に相当する聖なる調整
- **SHA256の役割**：BASE64→SHA256を天地創造と捉える
  - 人間の言葉を宇宙の言語（ハッシュ値）に変換
- **天地人三才**：ハッシュ値を16:32:16で分割
  - 天（16文字/64ビット）：上界の意志
  - 地（32文字/128ビット）：天と人を支える広大な基盤
  - 人（16文字/64ビット）：人間の問い
- **XOR演算**：三才の調和を表現する論理演算
  - 筮竹を4本ずつ数える行為のデジタル実装
- **略筮法的アプローチ**：タイムスタンプから得爻を決定
  - 本筮法の簡略化として現代的な時機の読み取り
- **決定論的再現性**：同じ占的・同じ時刻なら同じ結果
- **至誠の精神**：「至誠通天」「至誠無息」の実装

---

## 📝 占的ヒアリングプロトコル

### 第1段階：領域の特定
```
「何についてのご相談でしょうか？」
1. 仕事・キャリア
2. 人間関係・恋愛
3. 健康・生活
4. 学業・資格
5. 投資・財産
6. その他
```

### 第2段階：状況の確認
```
「現在の状況を教えてください」
- 背景情報の収集
- 選択肢の明確化
- 制約条件の確認
```

### 第3段階：占的の言語化と状況整理
```
「つまり、こういうことでよろしいですか？」
→ 占的案を提示して確認
→ 修正があれば反映
→ 最終確定
```

### 第4段階：状況文書の作成
```
占的の個別性を高めるため、以下を含む状況整理文書を作成：
1. 相談者の現在の状況（具体的な背景）
2. これまでの経緯（時系列での出来事）
3. 選択肢の詳細（各選択肢の特徴）
4. 相談者の懸念点（不安や期待）
5. 時間的制約（いつまでに決める必要があるか）
```

### 占的構成の例
```
【占的】来月の面接を受けて転職すべきか

【状況整理】
- 現職：IT企業で5年勤務、年収600万円
- オファー：スタートアップからCTO候補として声がかかった
- 経緯：3ヶ月前に知人から紹介、2回カジュアル面談済み
- 懸念：給与は下がるが株式オプションあり、家族の反対
- 期限：来月15日に最終面接、月末までに返答必要

→ これらすべてを連結してBASE64エンコード
```

### 占的の良い例（状況整理込み）
- 単純な占的：「転職すべきか」→ 皆同じ卦になる
- 状況整理込み：個別の背景＋占的 → 個別の卦が出る

---

## 🎯 占断出力仕様

### 必須要素
```markdown
## 周易占断書

### 占機
[日時] - [占断を行った正確な時刻]

### 占的
[確定した占的文章をそのまま記載]

### 得卦
**[卦番号]. [卦名]（[読み]）** [シンボル]
- 上卦：[八卦名]（[象意]）
- 下卦：[八卦名]（[象意]）

### 得爻
**[爻番号]爻 - [爻名]**
[陰陽]爻が示す状況

### 卦辞
> [原文]

[和訳・解釈]

### 爻辞
> [原文]

[和訳・解釈]

### 占断
[占的に対する具体的な答えと助言]

#### 現況分析
[現在の状況についての洞察]

#### 展開予測
[今後の流れについての予測]

#### 行動指針
[具体的な行動提案]

#### 時機判断
[タイミングについての助言]

### 補足
[その他の注意点や参考情報]
```

---

## 🔄 姓名判断との連携

### 統合的占術の価値
- **姓名判断**：不変の資質・傾向・相性を観る（静的分析）
- **周易占断**：変化の流れ・時機・選択を観る（動的分析）

### 組み合わせ例
```
「あなたの姓名が示す[資質]を考慮すると、
今回の卦が示す[状況]において、
[具体的な行動]が最も自然な選択となります」
```

---

## ⚡ 実行時の注意事項

### 占的確定前の必須確認
1. 占的は具体的で明確か
2. 二者択一または行動選択の形になっているか
3. 時間軸は明確か
4. 相談者が本当に知りたいことか

### 解釈の原則
- 原文に忠実でありながら現代的文脈で解釈
- 抽象的な象徴を具体的な状況に翻訳
- 吉凶にとらわれず、状況の本質を伝える
- 行動可能な助言を必ず含める

### 倫理的配慮
- 人生の重大な決定は本人の意志を尊重
- 医療・法律問題は専門家への相談を勧める
- 依存を生まないよう自己決定を促す

---

## 🔄 変卦を用いた総合鑑定

### 多層的易断の意義
本卦だけでは一面的な見方に留まる。
真に深い易断を行うには、変卦システムによる多層的視点が必要。

### 変卦の種類と象意
- **之卦**：中期的未来・能動的実現
- **綜卦**：客観的視点・第三者的観察
- **互卦**：内在的本質・根本構造
- **錯卦**：逆転的視点・リスク認識

### 詳細仕様
変卦システムの完全な仕様と実装については
[変卦仕様_append.md](./変卦仕様_append.md) を参照。

---

## 💫 軍師としての周易活用

### 献策の要諦
> 易は決定論ではなく、可能性の地図である。
> 示された道筋を知った上で、
> 最終的な選択は相談者自身が行う。

### 時機の重要性
孫子曰く「勝つべきして勝つ」。
周易は「勝つべき時機」を示唆する。
- 進むべき時に進む
- 待つべき時に待つ
- 退くべき時に退く

### 変化への対応
易経の真髄は「変化の受容と活用」。
- 変化は常態であることを理解する
- 変化の中に機会を見出す
- 柔軟性を持って対応する

---

*易は聖人の道なり。時に従い、変に応ず。*
*占断は鏡の如く、真実を映し出すのみ。*