# 変卦システム仕様 - Digital Shin-eki Extended
*デジタル心易システム仕様.md への追加仕様書*

---

## 🔄 変卦による多層的易断

### 変卦の意義
> 人間の運命は変転し続けて休まる所がない。
> 始まりがあれば終わりがあり、栄枯盛衰を繰り返すのが運命。
> その変転する運命を多面的に観るために、変卦を用いる。

本卦だけを観て判断するのは一面的な見方。奥行きのある易断を下すには、
本卦の之卦・綜卦・互卦・錯卦も考慮に入れた多層的な視野が必要となる。

---

## 📐 変卦の算出アルゴリズム

### 1. 之卦（しか）- 中期的な陰陽変化
```python
def calculate_zhigua(original_binary: str, line_number: int) -> str:
    """
    之卦：本卦の変爻を陰陽反転させた卦
    
    象意：中期的・実現・次の展開・能動
    意義：中期的な未来の運命を予見する
    
    Args:
        original_binary: 本卦の6桁バイナリ文字列（例："100001"）
        line_number: 変爻の番号（1-6、下から数える）
    
    Returns:
        之卦の6桁バイナリ文字列
    """
    binary_list = list(original_binary)
    # 爻は下から数えるので、配列インデックスは 6-line_number
    index = 6 - line_number
    binary_list[index] = '0' if binary_list[index] == '1' else '1'
    return ''.join(binary_list)

# 易断への応用
"""
本卦から之卦への変化は、運命の流れを示す。
例：乾為天九五 → 火天大有
「龍が天に昇るが如く好調を維持しながら、
後に大きな力や勢力を所有するであろう」
"""
```

### 2. 綜卦（そうけ）- 現実的な陰陽変化
```python
def calculate_zonggua(original_binary: str) -> str:
    """
    綜卦：本卦を180度回転（上下反転）させた卦
    
    象意：第三者的視点・客観的・全体・現実
    意義：運命を客観的に観る場合に用いる
    
    Args:
        original_binary: 本卦の6桁バイナリ文字列
    
    Returns:
        綜卦の6桁バイナリ文字列
    """
    return original_binary[::-1]

# 易断への応用
"""
自分から見る視点と、他人から見る視点が加わり、
全体像が掴めて現実的な判断を下すことが叶う。
例：沢水困 → 水風井
「困難は客観的に観れば自己修養であるから、
現在行っていることは的を射るので大いに努力しなさい」
"""
```

### 3. 互卦（ごけ）- 卦の本質
```python
def calculate_hugua(original_binary: str) -> str:
    """
    互卦：本卦の2,3,4爻を内卦、3,4,5爻を外卦とする卦
    
    象意：本質・中心・根本・理想
    意義：本卦の本質部分を観る
    
    Args:
        original_binary: 本卦の6桁バイナリ文字列
    
    Returns:
        互卦の6桁バイナリ文字列
    
    注：爻の位置（配列index対応）
    配列[0] = 第6爻（最上位）
    配列[1] = 第5爻
    配列[2] = 第4爻
    配列[3] = 第3爻
    配列[4] = 第2爻
    配列[5] = 第1爻（最下位）
    """
    # 2,3,4爻 = index[4,3,2]を下卦
    # 3,4,5爻 = index[3,2,1]を上卦
    lower = original_binary[4] + original_binary[3] + original_binary[2]
    upper = original_binary[3] + original_binary[2] + original_binary[1]
    return upper + lower

# 易断への応用
"""
勢いの弱い初爻と無位である上爻を除いた、
中心の爻である2,3,4,5爻から本質的な部分を観る。
例：地沢臨 → 地雷復
「行き詰っていた所から心機一転して現状を打破していく
臨機応変力が身についてくる」
"""
```

### 4. 錯卦（さくけ）- 陰陽の逆転的変化
```python
def calculate_cuogua(original_binary: str) -> str:
    """
    錯卦：本卦の全爻の陰陽を反転させた卦
    
    象意：逆転・逆説・裏側・悲観的
    意義：本卦の裏側を観る
    
    Args:
        original_binary: 本卦の6桁バイナリ文字列
    
    Returns:
        錯卦の6桁バイナリ文字列
    """
    return ''.join(['0' if bit == '1' else '1' for bit in original_binary])

# 易断への応用
"""
逆の方向に進んだ時はどうなるか、
物事が裏目に出たらどうするかという悲観的な目線を持って、
何を為すべきなのかを暗示する。
例：水火既済 → 火水未済
「物事が極まって完成を見ると、
その後には必ず崩壊するということを観る」
"""
```

---

## 🎯 変卦統合システム

### 完全実装クラス
```python
class DigitalHenkaSineki:
    """デジタル変卦システム - 多層的易断の実現"""
    
    def __init__(self, hexagram_database):
        self.db = hexagram_database
    
    def analyze_complete(self, original_binary: str, line_number: int) -> dict:
        """
        本卦から全ての変卦を算出し、統合的分析を行う
        
        Returns:
            {
                '本卦': {...},
                '之卦': {'binary': str, 'info': {...}, '象意': str},
                '綜卦': {'binary': str, 'info': {...}, '象意': str},
                '互卦': {'binary': str, 'info': {...}, '象意': str},
                '錯卦': {'binary': str, 'info': {...}, '象意': str},
                '統合解釈': str
            }
        """
        result = {
            '本卦': self.get_hexagram_info(original_binary)
        }
        
        # 各変卦の算出
        result['之卦'] = {
            'binary': self.calculate_zhigua(original_binary, line_number),
            '象意': '中期的未来・能動的実現・次の展開'
        }
        
        result['綜卦'] = {
            'binary': self.calculate_zonggua(original_binary),
            '象意': '客観的視点・第三者的観察・現実認識'
        }
        
        result['互卦'] = {
            'binary': self.calculate_hugua(original_binary),
            '象意': '内在的本質・中心構造・根本理想'
        }
        
        result['錯卦'] = {
            'binary': self.calculate_cuogua(original_binary),
            '象意': '逆転的視点・裏側観察・悲観的警告'
        }
        
        # 各変卦の情報取得
        for key in ['之卦', '綜卦', '互卦', '錯卦']:
            result[key]['info'] = self.get_hexagram_info(result[key]['binary'])
        
        # 統合解釈の生成
        result['統合解釈'] = self.generate_integrated_interpretation(result)
        
        return result
```

---

## 📊 変卦解釈マトリックス

### 変卦の相互関係と解釈指針

| 変卦種別 | 時間軸 | 視点 | 性質 | 易断での活用 |
|---------|--------|------|------|-------------|
| **本卦** | 現在 | 主観 | 表層 | 現状認識・直面する課題 |
| **之卦** | 中期的未来 | 能動 | 変化 | 展開予測・行動結果 |
| **綜卦** | 現在 | 客観 | 全体 | 他者視点・社会的評価 |
| **互卦** | 無時間 | 本質 | 深層 | 潜在構造・真の課題 |
| **錯卦** | 仮定的 | 逆説 | 裏側 | リスク認識・警告 |

### 統合的易断の実践

```markdown
## 多層的易断のテンプレート

### 1. 現状認識（本卦）
- 直面している状況の本質
- 表層的に現れている課題

### 2. 展開予測（本卦→之卦）
- 現在の行動がもたらす中期的結果
- 能動的に実現できる未来像

### 3. 客観的評価（綜卦）
- 第三者から見た現状の意味
- 社会的・全体的な視点での位置づけ

### 4. 本質洞察（互卦）
- 表層の背後にある真の構造
- 中心的・根本的な課題

### 5. リスク認識（錯卦）
- 逆方向に進んだ場合の結果
- 警戒すべき落とし穴

### 6. 統合的献策
- 5つの視点を統合した戦略的助言
- 時機を踏まえた具体的行動指針
```

---

## 🌟 実装例：山雷頤の変卦分析

```python
# オオイチョウタケ占断での実例
本卦 = "100001"  # 27.山雷頤
変爻 = 3  # 第3爻

変卦分析 = {
    "本卦": "27.山雷頤 - 養育の困難、栄養競合",
    "之卦": "21.火雷噬嗑 - 競合から協調への転換",
    "綜卦": "27.山雷頤 - 視点を変えても養育関係は不変",
    "互卦": "2.坤為地 - 土壌の豊かさは十分",
    "錯卦": "28.沢風大過 - KKKの過剰な勢い",
    
    "統合解釈": """
    現在は養分競合で苦戦（頤）しているが、
    将来的には共生関係を築ける可能性（噬嗑）がある。
    土壌自体は豊かで両種を養える力を持ち（坤）、
    KKKの過剰な勢いが落ち着けば（大過）、
    自然とバランスが回復する。
    
    【戦略的献策】
    競合ではなく共存を目指し、時間をかけた
    複合栽培システムの構築を推奨する。
    """
}
```

---

## 📝 運用上の注意事項

### 変卦解釈の原則

1. **相補的理解**
   - 変卦は互いに矛盾するものではなく、相補的な視点
   - 全ての変卦を総合して立体的な理解を構築

2. **文脈重視**
   - 占的の内容に応じて、重視する変卦を選択
   - 機械的な適用ではなく、柔軟な解釈を心がける

3. **時機の考慮**
   - 之卦は「いつ」実現するかも含めて解釈
   - 綜卦は「誰の視点か」を明確にして活用

4. **警告の活用**
   - 錯卦は単なる悲観ではなく、予防的指針として活用
   - リスクを認識した上での建設的な助言を導く

---

## 🎓 変卦システムの哲学

> 易は変化の書であり、変卦はその変化を多層的に観る技術。
> 一つの現象を五つの視点から観ることで、
> 運命の全体像を立体的に把握することが可能となる。

デジタル心易における変卦システムは、
古代の叡智とデジタル技術の融合により、
**再現性のある多層的易断**を実現する。

至誠通天の精神に基づき、誠実な問いに対して、
五つの視点から導かれる誠実な答えを返す。
それが、21世紀の変卦システムである。

---

*変化は常態、不変は幻想。*
*五つの視点が織りなす運命の地図を、誠実に読み解く。*